
// --------------------------------- LPShid.h -------------------------------------------
//
//	Include file for LabBrick digital phase shifter API
//
// (c) 2015 by Vaunix Corporation, all rights reserved
//
//	RD	derived from VNX_dps.h
//-----------------------------------------------------------------------------


#define VNX_MIN_DWELLTIME 1
#define VNX_PROFILE_LENGTH 50		// the profile can hold up to 50 elements

// Bit masks and equates for the Sweep command byte (stored in Sweep_mode, and reported also in Status)									
#define SWP_DIRECTION		0x04	// MASK: bit = 0 for sweep or ramp up, 1 for sweep or ramp down 
#define SWP_CONTINUOUS		0x02	// MASK: bit = 1 for continuous sweeping
#define SWP_ONCE		0x01		// MASK: bit = 1 for single sweep
#define SWP_BIDIR		0x10		// MASK: bit = 1 for triangle style sweep, 0 for ramp style
 
// HID report equates
#define HR_BLOCKSIZE 6				// size of the block of bytes buffer in our HID report
#define HID_REPORT_LENGTH 8 		// use an 8 byte report..

typedef struct 
{
  char reportid;
  char status;
  char count;
  char byteblock[HR_BLOCKSIZE];
} HID_REPORT1;

typedef struct 
{
  char reportid;
  char command;
  char count;
  char byteblock[HR_BLOCKSIZE];
} HID_REPORT_OUT;

// Misc commands to send to the device
// For the input reports the first byte is the status, for the output it is the command. The high bit sets the 
// direction.
//
//	count is the number of valid bytes in the byteblock array
// 	byteblock is an array of bytes which make up the value of the command's argument or arguments.
//
// For result reports the command portion of the status byte is equal to the command sent if the command was successful.
// status byte layout:

// Bit0 - Bit5 = command result, equal to command if everything went well
// Bit6 = --reserved--
// Bit7 = --reserved--

// All sweep related data items are DWORD (unsigned) quantities, stored in normal Microsoft byte order.
// Dwell time is a DWORD (unsigned)
//
// Many command types are shared by the DPS and other Lab Brick products, so constant names may reflect the product
// that the command was initially used for

#define VNX_SET				0x80
#define VNX_GET				0x00	// the set and get bits are or'd into the msb of the command byte

// -------------- Set/Get the phase angle, working frequency -----------------
#define VNX_PHASEANGLE		0x50	// phase shift angle in degrees (DWORD)
#define VNX_FREQUENCY		0x04	// working frequency in 100Khz units (set in 1MHz increments)

// ----------------- ramp commands ------------------------
#define VNX_SWEEP			0x09	// command to start/stop ramps, data = 01 for single ramp, 00 to stop
                                    // ramp, and 02 for continuous ramping.
#define VNX_ASTART			0x30	// initial value for the phase angle ramp
#define VNX_ASTOP			0x31	// final value for the phase angle ramp
#define VNX_ASTEP			0x32	// step size for the phase angle ramp
#define VNX_ADWELL			0x33	// dwell time for each phase angle step
#define VNX_AIDLE			0x36	// idle time between the phase angle ramps in milliseconds
#define VNX_ADWELL2			0x37	// dwell time for the second phase of the ramp
#define VNX_ASTEP2			0x38	// step size for the second phase of the ramp
#define VNX_AHOLD			0x39	// hold time between phase 1 and 2

#define VNX_SETPROFILE		0x3A	// set/get profile values, bytes 0 to 3 of the byteblock are interpreted as a DWORD
									// the phase angle is stored in the lower 2 bytes of the DWORD
									// the profile index is stored in byte 4 of the byteblock
#define VNX_PROFILECOUNT	0x3B	// number of elements in the profile, 1 to VNX_PROFILE_LENGTH = 50
#define VNX_PROFILEDWELL	0x3C	// dwell time for each profile element
#define VNX_PROFILEIDLE		0x3D	// idle time at the end of each repeating profile 

#define VNX_SAVEPAR			0x0C	// command to save user parameters to flash, data bytes must be
									// set to 0x42, 0x55, 0x31 as a key to enable the flash update
									// all of the settings are saved
									
#define VNX_GETSERNUM		0x1F	// get the serial number, value is a DWORD
#define VNX_MODELNAME		0x22	// get (no set allowed) the device's model name string -- last 6 chars only
#define VNX_DEFAULTS		0x0F	// restore all settings to factory default

#define VNX_MINATTEN		0x34	// get the minimum phase angle which is 0 for every LPS device now
#define VNX_MAXATTEN		0x35	// get the maximum phase angle which is 360 for every LPS device now
#define VNX_MINFREQUENCY	0x20	// get (no set allowed) the minimum operating frequency in 100KHz units
#define VNX_MAXFREQUENCY	0x21	// get (no set allowed) the maximum operating frequency in 100KHz units

//------------------------- Status Report ID Byte -------------------------------------
#define VNX_PHSTATUS		0x52	// Not really a command, but the status byte value for periodic status reports.


// ----------- Global Equates ------------
#define MAXDEVICES 64
#define MAX_MODELNAME 32

// ----------- Data Types ----------------

#define DEVID unsigned int

typedef struct
{
	int DevStatus;
	int Frequency;
	int MinFrequency;
	int MaxFrequency;
	int PhaseAngle;
	int MinPhase;
	int MaxPhase;
	int MinPhaseStep;
	int RampStart;
	int RampStop;
	int PhaseStep;
	int PhaseStep2;
	int DwellTime;
	int DwellTime2;
	int IdleTime;
	int HoldTime;
	int Modebits;
	int FrameNumber;
	int ProfileIndex;
	int ProfileDwellTime;
	int ProfileIdleTime;
	int ProfileCount;
	int SerialNumber;
	char ModelName[MAX_MODELNAME];
	// so we can find this device again later
	unsigned int idVendor;
	unsigned int idProduct;
	unsigned int idType;
	char Serialstr[16];
	char thread_command;
	char sndbuff[8];
	char rcvbuff[24];
	char decodewatch;
	int MyDevID;
	int Profile[50];
	bool NeedsProfileFix;
} LPSPARAMS;

// ----------- Mode Bit Masks ------------

#define MODE_RFON 0x00000010 	// bit is 1 for RF on, 0 if RF is off
#define MODE_INTREF 0x00000020 	// bit is 1 for internal osc., 0 for external reference
#define MODE_SWEEP 0x00000017 	// bottom 5 bits are used to keep the sweep control bits
								// bit 3 is the RF_ON bit, which is ignored, and hence zeroed out

// Bit masks and equates for the Ramp command byte (stored in Sweep_mode, and reported also in Status)
#define LPS_SWP_BIDIR			0x10	// MASK: bit = 0 for ramp style sweep, 1 for triangle style sweep
#define LPS_SWP_DIRECTION		0x04	// MASK: bit = 0 for sweep up, 1 for sweep down 
#define LPS_SWP_CONTINUOUS		0x02	// MASK: bit = 1 for continuous sweeping
#define LPS_SWP_ONCE			0x01	// MASK: bit = 1 for single sweep
#define STATUS_PROFILE_ACTIVE		0x80	// MASK: bit = 1 when a profile is active
#define LPS_PROFILE_ONCE		0x01	// play the profile once
#define LPS_PROFILE_REPEAT		0x02	// repeat the profile

// ----------- Command Equates -----------

// Status returns for commands
#define LPSTATUS unsigned int

#define STATUS_OK 			 0
#define BAD_PARAMETER		0x80010000	// out of range input -- parameter outside min/max etc.
#define BAD_HID_IO			0x80020000
#define DEVICE_NOT_READY	0x80030000	// device isn't open, no handle, etc.

// Status returns for DevStatus

#define INVALID_DEVID		0x80000000	// MSB is set if the device ID is invalid
#define DEV_CONNECTED		0x00000001	// LSB is set if a device is connected
#define DEV_OPENED			0x00000002	// set if the device is opened
#define SWP_ACTIVE			0x00000004	// set if the device is ramping
#define SWP_UP				0x00000008	// set if the device is ramping up
#define SWP_REPEAT			0x00000010	// set if the device is in continuous ramp mode
#define SWP_BIDIRECTIONAL	0x00000020	// set if the device is in bi-directional ramp mode
#define PROFILE_ACTIVE		0x00000040	// set if a profile is playing

// Internal values in DevStatus
#define DEV_LOCKED			0x00002000	// set if we don't want read thread updates of the device parameters
#define DEV_RDTHREAD		0x00004000	// set when the read thread is running

void fnLPS_Init(void);

void fnLPS_SetTestMode(bool testmode);
int fnLPS_GetNumDevices();
int fnLPS_GetDevInfo(DEVID *ActiveDevices);
int fnLPS_GetModelName(DEVID deviceID, char *ModelName);
LPSTATUS fnLPS_InitDevice(DEVID deviceID);
LPSTATUS fnLPS_CloseDevice(DEVID deviceID);
int fnLPS_GetSerialNumber(DEVID deviceID);
char* fnLPS_LibVersion(void);

LPSTATUS fnLPS_SetPhaseAngle(DEVID deviceID, int phase);
LPSTATUS fnLPS_SetWorkingFrequency(DEVID deviceID, int frequency);
LPSTATUS fnLPS_SetRampStart(DEVID deviceID, int rampstart);
LPSTATUS fnLPS_SetRampEnd(DEVID deviceID, int rampstop);
LPSTATUS fnLPS_SetPhaseAngleStep(DEVID deviceID, int phasestep);
LPSTATUS fnLPS_SetPhaseAngleStepTwo(DEVID deviceID, int phasestep2);
LPSTATUS fnLPS_SetDwellTime(DEVID deviceID, int dwelltime);
LPSTATUS fnLPS_SetDwellTimeTwo(DEVID deviceID, int dwelltime2);
LPSTATUS fnLPS_SetIdleTime(DEVID deviceID, int idletime);
LPSTATUS fnLPS_SetHoldTime(DEVID deviceID, int holdtime);

LPSTATUS fnLPS_SetProfileElement(DEVID deviceID, int index, int phaseangle);
LPSTATUS fnLPS_SetProfileCount(DEVID deviceID, int profilecount);
LPSTATUS fnLPS_SetProfileIdleTime(DEVID deviceID, int idletime);
LPSTATUS fnLPS_SetProfileDwellTime(DEVID deviceID, int dwelltime);
LPSTATUS fnLPS_StartProfile(DEVID deviceID, int mode);

LPSTATUS fnLPS_SetRampDirection(DEVID deviceID, bool up);
LPSTATUS fnLPS_SetRampMode(DEVID deviceID, bool mode);
LPSTATUS fnLPS_SetRampBidirectional(DEVID deviceID, bool bidir_enable);
LPSTATUS fnLPS_StartRamp(DEVID deviceID, bool go);

LPSTATUS fnLPS_SaveSettings(DEVID deviceID);

int fnLPS_GetPhaseAngle(DEVID deviceID);
int fnLPS_GetWorkingFrequency(DEVID deviceID);

int fnLPS_GetRampStart(DEVID deviceID);
int fnLPS_GetRampEnd(DEVID deviceID);
int fnLPS_GetDwellTime(DEVID deviceID);
int fnLPS_GetDwellTimeTwo(DEVID deviceID);
int fnLPS_GetIdleTime(DEVID deviceID);
int fnLPS_GetHoldTime(DEVID deviceID);

int fnLPS_GetPhaseAngleStep(DEVID deviceID);
int fnLPS_GetPhaseAngleStepTwo(DEVID deviceID);

int fnLPS_GetProfileElement(DEVID deviceID, int index);
int fnLPS_GetProfileCount(DEVID deviceID);
int fnLPS_GetProfileDwellTime(DEVID deviceID);
int fnLPS_GetProfileIdleTime(DEVID deviceID);
int fnLPS_GetProfileIndex(DEVID deviceID);

int fnLPS_GetMaxPhaseShift(DEVID deviceID);
int fnLPS_GetMinPhaseShift(DEVID deviceID);
int fnLPS_GetMinPhaseStep(DEVID deviceID);

int fnLPS_GetMaxWorkingFrequency(DEVID deviceID);
int fnLPS_GetMinWorkingFrequency(DEVID deviceID);

LPSTATUS fnLPS_SetRFOn(DEVID deviceID, bool on);	// ignored by LPS hardware, always on
int fnLPS_GetRF_On(DEVID deviceID);					// always true for a LPS-802, 402, 123 or 202 device

char* fnLPS_perror(LPSTATUS status);

